/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.coordination;

import com.datastax.driver.core.exceptions.NoHostAvailableException;
import com.datastax.driver.core.exceptions.QueryConsistencyException;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.EventBus;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.common.server.utils.MillisecondBoundConfiguration;
import org.apache.cassandra.sidecar.common.server.utils.SecondBoundConfiguration;
import org.apache.cassandra.sidecar.config.ClusterLeaseClaimConfiguration;
import org.apache.cassandra.sidecar.config.ServiceConfiguration;
import org.apache.cassandra.sidecar.coordination.ClusterLease;
import org.apache.cassandra.sidecar.coordination.ElectorateMembership;
import org.apache.cassandra.sidecar.db.SidecarLeaseDatabaseAccessor;
import org.apache.cassandra.sidecar.metrics.DeltaGauge;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.metrics.server.CoordinationMetrics;
import org.apache.cassandra.sidecar.server.SidecarServerEvents;
import org.apache.cassandra.sidecar.tasks.PeriodicTask;
import org.apache.cassandra.sidecar.tasks.PeriodicTaskExecutor;
import org.apache.cassandra.sidecar.tasks.ScheduleDecision;
import org.apache.cassandra.sidecar.utils.EventBusUtils;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterLeaseClaimTask
implements PeriodicTask {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClusterLeaseClaimTask.class);
    static final SecondBoundConfiguration MINIMUM_DELAY = SecondBoundConfiguration.parse((String)"30s");
    private final ElectorateMembership electorateMembership;
    private final SidecarLeaseDatabaseAccessor accessor;
    private final ClusterLease clusterLease;
    private final CoordinationMetrics metrics;
    private final ClusterLeaseClaimConfiguration configuration;
    private final ServiceConfiguration config;
    private EventBus eventBus;
    private String currentLeaseholder;
    private Instant leaseTime;

    public ClusterLeaseClaimTask(ServiceConfiguration serviceConfiguration, ElectorateMembership electorateMembership, SidecarLeaseDatabaseAccessor accessor, ClusterLease clusterLease, SidecarMetrics metrics) {
        this.configuration = serviceConfiguration.coordinationConfiguration().clusterLeaseClaimConfiguration();
        this.config = serviceConfiguration;
        this.electorateMembership = electorateMembership;
        this.accessor = accessor;
        this.clusterLease = clusterLease;
        this.metrics = metrics.server().coordination();
    }

    @Override
    public void deploy(Vertx vertx, PeriodicTaskExecutor executor) {
        this.eventBus = vertx.eventBus();
        EventBusUtils.onceLocalConsumer(this.eventBus, SidecarServerEvents.ON_CASSANDRA_CQL_READY.address(), ignored -> executor.schedule(this));
    }

    @Override
    public ScheduleDecision scheduleDecision() {
        boolean isEnabled = this.config.schemaKeyspaceConfiguration().isEnabled() && this.configuration.enabled();
        boolean isMember = false;
        if (isEnabled) {
            try {
                isMember = this.electorateMembership.isMember();
            }
            catch (Throwable t) {
                LOGGER.debug("Membership determination fails due to unexpected exception", t);
            }
            LOGGER.debug("Sidecar instance part of electorate isMember={}", (Object)isMember);
        }
        if (!isEnabled || !isMember) {
            this.clusterLease.setOwnership(ClusterLease.Ownership.LOST);
            return ScheduleDecision.SKIP;
        }
        return this.accessor.isAvailable() ? ScheduleDecision.EXECUTE : ScheduleDecision.RESCHEDULE;
    }

    @Override
    public DurationSpec initialDelay() {
        long randomDeltaDelayMillis = this.configuration.randomDeltaDelayMillis();
        if (randomDeltaDelayMillis == 0L) {
            return this.configuration.initialDelay();
        }
        long initialDelayMillis = this.configuration.initialDelay().toMillis() + randomDeltaDelayMillis;
        return new MillisecondBoundConfiguration(initialDelayMillis, TimeUnit.MILLISECONDS);
    }

    @Override
    public DurationSpec delay() {
        MillisecondBoundConfiguration delay = this.configuration.executeInterval();
        if (delay.compareTo((DurationSpec)MINIMUM_DELAY) < 0) {
            LOGGER.warn("Ignoring delay value {} which is less than the required minimum of {}", (Object)delay, (Object)MINIMUM_DELAY);
            return MINIMUM_DELAY;
        }
        return delay;
    }

    @Override
    public void execute(Promise<Void> promise) {
        this.runClaimProcess();
        promise.complete();
    }

    @VisibleForTesting
    protected void runClaimProcess() {
        String sidecarHostId = this.sidecarHostId();
        boolean wasLeaseholder = this.isCurrentLeaseholder(sidecarHostId);
        this.tryToClaimClusterLease(sidecarHostId);
        this.updateClusterLease(sidecarHostId, wasLeaseholder);
        this.maybeNotify(sidecarHostId, wasLeaseholder);
        this.updateMetrics();
    }

    protected void tryToClaimClusterLease(String sidecarHostId) {
        LOGGER.debug("Starting selection for sidecarHostId={}", (Object)sidecarHostId);
        this.currentLeaseholder = this.isCurrentLeaseholder(sidecarHostId) ? this.executeLeaseAction("extend", sidecarHostId, this.accessor::extendLease) : this.executeLeaseAction("claim", sidecarHostId, this.accessor::claimLease);
    }

    protected String executeLeaseAction(String actionName, String sidecarHostId, Function<String, SidecarLeaseDatabaseAccessor.LeaseClaimResult> actionFn) {
        try {
            LOGGER.debug("Attempting to {} lease for sidecarHostId={}", (Object)actionName, (Object)sidecarHostId);
            return actionFn.apply((String)sidecarHostId).currentOwner;
        }
        catch (NoHostAvailableException | QueryConsistencyException | IllegalArgumentException e) {
            LOGGER.debug("Unable to {} lease for sidecarHostId={}", new Object[]{actionName, sidecarHostId, e});
        }
        catch (Exception e) {
            LOGGER.error("Unable to {} lease for sidecarHostId={}", new Object[]{actionName, sidecarHostId, e});
        }
        return null;
    }

    protected void updateClusterLease(String sidecarHostId, boolean wasLeaseholder) {
        if (this.currentLeaseholder == null) {
            boolean leaseExpired = this.leaseExpired();
            if (wasLeaseholder && !leaseExpired) {
                this.currentLeaseholder = sidecarHostId;
                LOGGER.debug("Lease will expire on {}. Assume the current leaseholder, even though no leaseholder is resolved from this run", (Object)this.leaseExpirationTime());
            } else {
                if (leaseExpired) {
                    LOGGER.info("Giving up lease for sidecarHostId={} leaseAcquired={} leaseExpired={}", new Object[]{sidecarHostId, this.leaseTime, this.leaseExpirationTime()});
                }
                this.leaseTime = null;
                this.clusterLease.setOwnership(ClusterLease.Ownership.INDETERMINATE);
            }
            return;
        }
        if (this.isCurrentLeaseholder(sidecarHostId)) {
            this.leaseTime = Instant.now();
            this.clusterLease.setOwnership(ClusterLease.Ownership.CLAIMED);
        } else {
            this.leaseTime = null;
            this.clusterLease.setOwnership(ClusterLease.Ownership.LOST);
        }
    }

    protected void maybeNotify(String sidecarHostId, boolean wasLeaseholder) {
        boolean isCurrentLeaseholder = this.isCurrentLeaseholder(sidecarHostId);
        if (wasLeaseholder && !isCurrentLeaseholder) {
            LOGGER.info("Cluster-wide lease has been lost by sidecarHostId={}", (Object)sidecarHostId);
            this.eventBus.publish(SidecarServerEvents.ON_SIDECAR_GLOBAL_LEASE_LOST.address(), (Object)sidecarHostId);
        }
        if (!wasLeaseholder && isCurrentLeaseholder) {
            LOGGER.info("Cluster-wide lease has been claimed by sidecarHostId={}", (Object)sidecarHostId);
            this.eventBus.publish(SidecarServerEvents.ON_SIDECAR_GLOBAL_LEASE_CLAIMED.address(), (Object)sidecarHostId);
        }
        if (LOGGER.isDebugEnabled() && wasLeaseholder && isCurrentLeaseholder) {
            LOGGER.debug("Cluster-wide lease has been extended by sidecarHostId={}", (Object)sidecarHostId);
        }
    }

    void updateMetrics() {
        if (this.clusterLease.isClaimedByLocalSidecar()) {
            ((DeltaGauge)this.metrics.leaseholders.metric).update(1L);
        }
        ((DeltaGauge)this.metrics.participants.metric).update(1L);
    }

    private boolean leaseExpired() {
        return this.leaseTime != null && this.leaseExpirationTime().isBefore(Instant.now());
    }

    private Instant leaseExpirationTime() {
        return this.leaseTime.plus(this.config.schemaKeyspaceConfiguration().leaseSchemaTTL().toSeconds(), ChronoUnit.SECONDS);
    }

    protected String sidecarHostId() {
        return this.config.hostId();
    }

    private boolean isCurrentLeaseholder(String sidecarHostId) {
        return Objects.equals(sidecarHostId, this.currentLeaseholder);
    }

    @VisibleForTesting
    void resetLeaseholder() {
        this.currentLeaseholder = null;
        this.leaseTime = null;
        this.clusterLease.setOwnership(ClusterLease.Ownership.INDETERMINATE);
    }
}

