/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.sidecar.handlers;

import java.util.Collections;
import java.util.Set;

import com.google.inject.Inject;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.net.SocketAddress;
import io.vertx.ext.auth.authorization.Authorization;
import io.vertx.ext.auth.authorization.OrAuthorization;
import io.vertx.ext.web.RoutingContext;
import org.apache.cassandra.sidecar.acl.authorization.BasicPermissions;
import org.apache.cassandra.sidecar.common.server.MetricsOperations;
import org.apache.cassandra.sidecar.common.server.data.QualifiedTableName;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.utils.CassandraInputValidator;
import org.apache.cassandra.sidecar.utils.InstanceMetadataFetcher;

/**
 * Handler for retrieving table stats
 */
public class TableStatsHandler extends AbstractHandler<QualifiedTableName> implements AccessProtected
{

    /**
     * Constructs a handler with the provided {@code metadataFetcher}
     *
     * @param metadataFetcher the metadata fetcher
     * @param executorPools   executor pools for blocking executions
     */
    @Inject
    protected TableStatsHandler(InstanceMetadataFetcher metadataFetcher,
                                ExecutorPools executorPools,
                                CassandraInputValidator validator)
    {
        super(metadataFetcher, executorPools, validator);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Set<Authorization> requiredAuthorizations()
    {
        OrAuthorization authorization = OrAuthorization.create();
        authorization.addAuthorization(BasicPermissions.STATS_TABLE_SCOPED.toAuthorization());
        authorization.addAuthorization(BasicPermissions.STATS_CLUSTER_SCOPED.toAuthorization());
        return Collections.singleton(authorization);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void handleInternal(RoutingContext context, HttpServerRequest httpRequest, String host, SocketAddress remoteAddress, QualifiedTableName tableName)
    {
        MetricsOperations operations = metadataFetcher.delegate(host).metricsOperations();
        executorPools.service()
                     .executeBlocking(() -> operations.tableStats(tableName))
                     .onSuccess(context::json)
                     .onFailure(cause -> processFailure(cause, context, host, remoteAddress, tableName));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected QualifiedTableName extractParamsOrThrow(RoutingContext context)
    {
        return qualifiedTableName(context);
    }
}
